/*
 *	Handle incoming frames
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"
#include <linux/inetdevice.h>
#include <linux/udp.h>
#include <linux/tcp.h>
#include <linux/ip.h>
#include <linux/if_ether.h>
#include <linux/netlink.h>
#include <linux/skbuff.h>


/* Bridge group multicast address 802.1d (pg 51). */
const u8 br_group_address[ETH_ALEN] = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x00 };

#ifdef CONFIG_DNS_HTTP_HIJACK
#include <net/ip.h>
struct net_bridge_port *pbr0 = NULL;
int sysctl_dns_hijack; //dns hijack switch
enum {
    BLANK_STATE = 0x02,
    WDS_REPEATER = 0x04
};
#ifndef CONFIG_APPS_LOGIC_APCLIENT
#if defined (CONFIG_RALINK_RT6855) || defined (CONFIG_RALINK_RT6855A) || \
    defined (CONFIG_RALINK_MT7620)|| defined (CONFIG_RALINK_MT7621)
//WAN·ͨʶ
int g_wan_intf_up = 0;
//switchݶ˿ڵԴ״̬޸g_wan_intf_upd
EXPORT_SYMBOL(g_wan_intf_up);
#endif
#endif

//default is 192.168.1.1
__be32 sysctl_hijack_ip = 0x6600A8C0; 

unsigned char sysctl_hijack_ipv6[16] = 
{
	0xfe,0x80,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
};

struct dns_header {
    unsigned short 	id;
    unsigned short  flags;

    unsigned short	qdcount; //question count
    unsigned short	ancount; //answer counet
    unsigned short	nscount;
    unsigned short	arcount;	/* Till here it would fit perfectly to a real DNS packet if we had big endian. */
};

static int dns_query_name_sizeof(unsigned char *name)
{
	int size = 0;
	if(NULL == name)
		return size;
	
	while(name[size++] != 0);
	return size;	
}

#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)

static int br_ipv6_filter_dns_packet(struct sk_buff *skb,u_int32_t  dns_ttl,int aaaa_flag)
{
	struct sk_buff *newskb = NULL;
	struct ethhdr *eth = NULL;
	unsigned char tmpmac[ETH_ALEN] = {0};
	struct ipv6hdr *iph = (struct ipv6hdr *)skb_network_header(skb), *newiph = NULL;
	struct udphdr *udph = (void *)iph + sizeof(struct ipv6hdr), *newudph = NULL;
	u_int32_t udplen, newlen, newudplen;
	struct in6_addr  tmpip;
	__be16 tmpport = 0;
	u_int32_t maxQueryLen = 0;
	u_int32_t queryLen = 0;
	// caculate new UDP packet length
	unsigned int size = 0;
	struct dns_header *dns_hdr =(struct dns_header *)((unsigned char *)udph + sizeof(struct udphdr));
	unsigned char *dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
	queryLen = dns_query_name_sizeof((unsigned char *)dns_query);
	if(aaaa_flag == 1)
		size = queryLen + 10 +  16;
	else
		size = queryLen + 14;

	udplen = skb->len - sizeof(struct ipv6hdr);

	maxQueryLen = udplen - sizeof(struct udphdr) - sizeof(struct dns_header) - 4;

	//printk("###############%s %d maxQueryLen:%d queryLen:%d\n",__FUNCTION__,__LINE__,maxQueryLen,queryLen);
	//ѯȵĴСӦСڻmaxQueryLen
	if(queryLen > maxQueryLen)
	{
		return 0;
	}

	newudplen = udplen + size;
	newlen = sizeof(struct ipv6hdr) + newudplen;

	skb_push(skb, ETH_HLEN);
	newskb = skb_copy_expand(skb, skb_headroom(skb), size, GFP_ATOMIC);
	skb_pull(skb, ETH_HLEN);

	if (NULL == newskb)
		return 0;
	
	skb_put(newskb, size);

	// reverse destinate MAC and source MAC
	eth = eth_hdr(newskb);
	memcpy(tmpmac, eth->h_source, ETH_ALEN);
	memcpy(eth->h_source, eth->h_dest, ETH_ALEN);
	memcpy(eth->h_dest, tmpmac, ETH_ALEN);

	// reverse destinate IP and source IP
	newiph = (struct ipv6hdr *)skb_network_header(newskb);
	tmpip = newiph->saddr;
	newiph->saddr = newiph->daddr;
	newiph->daddr = tmpip;

	newiph->payload_len = htons(newudplen);

	// modify UDP packet values
	newudph = (void *)newiph + sizeof(struct ipv6hdr);
	tmpport = newudph->source;
	newudph->source = newudph->dest;
	newudph->dest = tmpport;
	newudph->len = htons(newudplen);

	
	// modify DNS packet values
	dns_hdr = (struct dns_header *)((unsigned char *)newudph + sizeof(struct udphdr));
	dns_hdr->flags = htons(0x8180); // standard query response, no error
	dns_hdr->ancount = htons(1);
	
	{
		unsigned int *tol;
		unsigned short *dl;
		
		unsigned int query_size = 0;
		if(aaaa_flag == 1)
			query_size = size - 22;
		else	
			query_size = size - 10;
		dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
		unsigned char *dns_response = (unsigned char *)dns_query + query_size;	
		if(aaaa_flag == 1)
			memcpy(dns_response, dns_query, size - 12);
		else	
			memcpy(dns_response, dns_query, size);
		tol = (unsigned int *)(dns_response + query_size);
		//DNSЧʱ
		*tol = htonl(dns_ttl);
		dl = (unsigned short *)(dns_response + query_size + 4);
		if(aaaa_flag == 1)
		{
		/*
			unsigned char  *ip;
			*dl = htons(16);
			ip = (unsigned char  *)(dns_response + query_size + 6);
			//*ip = newiph->saddr;
			memcpy(ip ,sysctl_hijack_ipv6,16); //force to the LAN Manage IPv6
			*/
			//AAAA¼ֱӷ
			return 1;
		}else
		{
			__be32 *ip;
			*dl = htons(4);
			ip = (__be32 *)(dns_response + query_size + 6);
			//*ip = newiph->saddr;
			*ip = (__be32)(sysctl_hijack_ip); //force to the LAN Manage IP			
		}
	}


	/* fix udp checksum if udp checksum was previously calculated */
	if (newudph->check != 0) {
		newudph->check = 0;
		newudph->check = csum_ipv6_magic(&newiph->saddr, &newiph->daddr, newudplen,
				     IPPROTO_UDP, csum_partial((char *)newudph,
						             newudplen, 0));		
	}

	if(newskb->dev != NULL && newskb->dev->netdev_ops != NULL &&
	newskb->dev->netdev_ops->ndo_start_xmit != NULL)
	{
	       //printk("\n++++++++++++++++++++++ndo_start_xmit DNS Response+++++++++++++++++++++\n");
		newskb->dev->netdev_ops->ndo_start_xmit(newskb, newskb->dev);
	}
	else
	{	    		
#if 0		
	   if(printk_ratelimit())
		{
			printk("\n++++++++++++++++++++++ndo_start_xmit DNS error +++++++++++++++++++++\n");
		}
#endif
		kfree_skb(newskb);	
	}
	
	return 1;
	
}
#endif
static int br_filter_dns_packet(struct sk_buff *skb,struct net_bridge_port *p,u_int32_t  dns_ttl)
{
	struct sk_buff *newskb = NULL;
	struct ethhdr *eth = NULL;
	unsigned char tmpmac[ETH_ALEN] = {0};
	struct iphdr *iph = (struct iphdr *)skb_network_header(skb), *newiph = NULL;
	struct udphdr *udph = (void *)iph + iph->ihl * 4, *newudph = NULL;
	u_int32_t udplen, newlen, newudplen;
	__be32 tmpip = 0;
	__be16 tmpport = 0;
	u_int32_t maxQueryLen = 0;
	u_int32_t queryLen = 0;

	// caculate new UDP packet length
	unsigned int size = 0;
	struct dns_header *dns_hdr =(struct dns_header *)((unsigned char *)udph + sizeof(struct udphdr));
	unsigned char *dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
	queryLen = dns_query_name_sizeof((unsigned char *)dns_query);
	size = queryLen + 14;

	udplen = skb->len - iph->ihl*4;

	maxQueryLen = udplen - sizeof(struct udphdr) - sizeof(struct dns_header) - 4;

	//printk("###############%s %d maxQueryLen:%d queryLen:%d\n",__FUNCTION__,__LINE__,maxQueryLen,queryLen);
	//ѯȵĴСӦСڻmaxQueryLen
	if(queryLen > maxQueryLen)
	{
		return 0;
	}


	newudplen = udplen + size;
	newlen = iph->ihl*4 + newudplen;

	skb_push(skb, ETH_HLEN);
	newskb = skb_copy_expand(skb, skb_headroom(skb), size, GFP_ATOMIC);
	skb_pull(skb, ETH_HLEN);

	if (NULL == newskb)
		return 0;

	//newiph = (struct iphdr *)skb_network_header(newskb);
	//newudph = (void *)newiph + newiph->ihl*4;
	skb_put(newskb, size);

	// reverse destinate MAC and source MAC
	eth = eth_hdr(newskb);
	memcpy(tmpmac, eth->h_source, ETH_ALEN);
	memcpy(eth->h_source, eth->h_dest, ETH_ALEN);
	memcpy(eth->h_dest, tmpmac, ETH_ALEN);

	// reverse destinate IP and source IP
	newiph = (struct iphdr *)skb_network_header(newskb);
	tmpip = newiph->saddr;
	newiph->saddr = newiph->daddr;
	newiph->daddr = tmpip;

	// modify the rest of IP packet values
	newiph->tot_len = htons(newlen);
	newiph->id = 0;
	newiph->frag_off = htons(0x4000); // do not fragment
	newiph->ttl = 64;
	
	// modify UDP packet values
	newudph = (void *)newiph + newiph->ihl*4;
	tmpport = newudph->source;
	newudph->source = newudph->dest;
	newudph->dest = tmpport;
	newudph->len = htons(newudplen);

	// modify DNS packet values
	dns_hdr = (struct dns_header *)((unsigned char *)newudph + sizeof(struct udphdr));
	dns_hdr->flags = htons(0x8180); // standard query response, no error
	dns_hdr->ancount = htons(1);

	// add DNS answer
	{
		unsigned int *tol;
		unsigned short *dl;
		__be32 *ip;
		unsigned int query_size = size - 10;
		dns_query = (unsigned char *)dns_hdr + sizeof(struct dns_header);
		unsigned char *dns_response = (unsigned char *)dns_query + query_size;	
		memcpy(dns_response, dns_query, size);
		tol = (unsigned int *)(dns_response + query_size);
		//DNSЧʱ
		*tol = htonl(dns_ttl);
		dl = (unsigned short *)(dns_response + query_size + 4);
		*dl = htons(4);
		ip = (__be32 *)(dns_response + query_size + 6);
		//*ip = newiph->saddr;
#if 0
		/*ϻȡip */
		struct in_ifaddr *ifa;
		struct in_device *in_dev = __in_dev_get_rcu(pbr0->dev);
		if (!in_dev)
			return 1;
		printk("\n++++++file: %s, function: %s, line: %d, dev: %s  +++++++++\n",__FILE__,__FUNCTION__,__LINE__,skb->dev->name);
		for (ifa = in_dev->ifa_list; ifa; ifa = ifa->ifa_next){
			printk("\n++++ file: %s, function: %s, line: %d, dev: %s +++++\n", __FILE__,__FUNCTION__,__LINE__,ifa->ifa_label);
			if (strcmp(ifa->ifa_label, "br0") == 0){
				*ip = (__be32)ifa->ifa_local;
				printk("\n++++ ifle: %s, function: %s, line: %d, ifa_local: %x, ifa_address: %x +++++\n", __FILE__,__FUNCTION__,__LINE__,ifa->ifa_local,ifa->ifa_address);
			}
		}
#endif	
		*ip = (__be32)(sysctl_hijack_ip); //force to the LAN Manage IP
	}

	/* fix udp checksum if udp checksum was previously calculated */
	if (newudph->check != 0) {
		newudph->check = 0;
		newudph->check = csum_tcpudp_magic(newiph->saddr, newiph->daddr,
						newudplen, IPPROTO_UDP,
						csum_partial((char *)newudph,
						             newudplen, 0));
	}
	ip_send_check(newiph); //ip checksum

//printk("\n++++++file: %s, function: %s, line: %d +++++++++\n",__FILE__,__FUNCTION__,__LINE__);
	if(newskb->dev != NULL && newskb->dev->netdev_ops != NULL &&
                newskb->dev->netdev_ops->ndo_start_xmit != NULL)
    {
	   // printk("\n++++++++++++++++++++++ndo_start_xmit DNS Response+++++++++++++++++++++\n");
	    newskb->dev->netdev_ops->ndo_start_xmit(newskb, newskb->dev);
    }
	else
	{	    		
#if 0		
	   if(printk_ratelimit())
		{
			printk("\n++++++++++++++++++++++ndo_start_xmit DNS error +++++++++++++++++++++\n");
		}
#endif
		kfree_skb(newskb);	
	}
	return 1;

}

static int findstring(char * source, char * dest,int length,int cmplen)
{
	if(source == NULL || dest == NULL )
		return NULL;
	char *s1 = source, *d1 = dest;
	int i =0 ;
	while ( i < (length - cmplen) ) //ַȽϳȣԷֹԽ
	{
		if ( memcmp(s1,d1,cmplen) == 0)
			return 1;
		s1 = ++source;
		i++;
	}
	return 0;
}

//DNSЧʱ
#define DNS_TTL 	0		
#if 0
static const unsigned char router_domain0[] = {0x3,'w','w','w',0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'c','o','m',0};
static const unsigned char router_domain1[] = {0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'c','o','m',0};
static const unsigned char router_domain2[] = {0x3,'w','w','w',0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'n','e','t',0};
static const unsigned char router_domain3[] = {0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'n','e','t',0};
static const unsigned char router_domain4[] = {0x0a,'r','e','a','d','y','s','h','a','r','e',0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'n','e','t',0};
static const unsigned char router_domain5[] = {0x0a,'r','e','a','d','y','s','h','a','r','e',0x0b,'r','o','u','t','e','r','l','o','g','i','n',0x3,'c','o','m',0};
#else
static const unsigned char router_domain0[] = {0x3,'w','w','w',0x09,'m','y','w','i','f','i','e','x','t',0x3,'n','e','t',0};
static const unsigned char router_domain1[] = {0x3,'w','w','w',0x0f,'n','e','t','g','e','a','r','e','x','t','e','n','d','e','r',0x3,'n','e','t',0};
static const unsigned char router_domain2[] = {0x3,'w','w','w',0x09,'m','y','w','i','f','i','e','x','t',0x3,'c','o','m',0};
static const unsigned char router_domain3[] = {0x3,'w','w','w',0x0f,'n','e','t','g','e','a','r','e','x','t','e','n','d','e','r',0x3,'c','o','m',0};
static const unsigned char router_domain4[] = {0x09,'m','y','w','i','f','i','e','x','t',0x3,'n','e','t',0};
static const unsigned char router_domain5[] = {0x0f,'n','e','t','g','e','a','r','e','x','t','e','n','d','e','r',0x3,'n','e','t',0};
static const unsigned char router_domain6[] = {0x09,'m','y','w','i','f','i','e','x','t',0x3,'c','o','m',0};
static const unsigned char router_domain7[] = {0x0f,'n','e','t','g','e','a','r','e','x','t','e','n','d','e','r',0x3,'c','o','m',0};
#endif
static const unsigned char xbox_domain0[] = {0x2,'a','s',0x8,'x','b','o','x','l','i','v','e',0x3,'c','o','m',0};
static const unsigned char xbox_domain1[] = {0x3,'t','g','s',0x8,'x','b','o','x','l','i','v','e',0x3,'c','o','m',0};
static const unsigned char xbox_domain2[] = {0x4,'m','a','c','s',0x8,'x','b','o','x','l','i','v','e',0x3,'c','o','m',0};

//www.routerlogin.com
//routerlogin.com
//www.routerlogin.net
//routerlogin.net
//readyshare.routerlogin.net
//readyshare.routerlogin.com
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)

int br_ipv6_dns_netgear(struct sk_buff * skb)
{
	struct ipv6hdr *hdr;
	struct udphdr                   *udph;
	unsigned char *ptmp = NULL;
	struct dns_header *dns_hdr;
	unsigned short type = 0;
	int querySize = 0;
	hdr = (struct ipv6hdr *) skb->data;
	if(hdr->nexthdr != IPPROTO_UDP)
		{
		return 1;
		}
	
	udph = (struct udphdr *)(((unsigned char *)hdr) + sizeof(struct ipv6hdr));

	if( udph->dest != htons(53) )
		{
		return 1;
		}
	ptmp = (unsigned char *)udph + sizeof(struct udphdr);
	dns_hdr = (struct dns_header *)ptmp;

	//flags is query, and the query count is equal 1
    	if (((dns_hdr->flags & htons(0x8000)) != 0)  || (dns_hdr->qdcount != htons(1))) 
    		{
		return 1;
    		}
	
	
	ptmp = (unsigned char *)dns_hdr + sizeof(struct dns_header);
	querySize = dns_query_name_sizeof(ptmp);
	ptmp += querySize;
	type = *((unsigned short *)ptmp);	//get the  query type	

	//DNSʽʾķDNSĺԴ
	if(ptmp + 4 > skb->tail)
	{
		return 1;
	}
	
	/* 
	AAAAͲѯ֧֣ӦĿֹֹ
	AAAAͲѯ
	*/
	if((type != htons(0x0001)) &&  (type != htons(0x001c)))//"Host Address"query
	{
		return 1;
	}
	if(findstring((unsigned char *)((unsigned char *)udph + 12),xbox_domain0,udph->len - 12, sizeof(xbox_domain0)-1)
	||findstring((unsigned char *)((unsigned char *)udph + 12),xbox_domain1,udph->len - 12, sizeof(xbox_domain1)-1)
	||findstring((unsigned char *)((unsigned char *)udph + 12),xbox_domain2,udph->len - 12, sizeof(xbox_domain2)-1)
	/*WDS Repeater Modeٳֹ*/
	|| (WDS_REPEATER&sysctl_dns_hijack))
	{
		//صҪ
		return 1;
	}
	/*жǷΪBlank Stateδ״̬*/
	else if (BLANK_STATE&sysctl_dns_hijack)
	{
		br_ipv6_filter_dns_packet(skb,0,(type == htons(0x001c)) ? 1:0); //construct the dns replay packet
		return 0;
	}
	//ұӦDNSѯЧʱΪһ
	else if(findstring((unsigned char *)((unsigned char *)udph + 12),router_domain0,udph->len - 12, sizeof(router_domain0)-1)
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain1,udph->len - 12, sizeof(router_domain1)-1) 
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain2,udph->len - 12, sizeof(router_domain2)-1) 
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain3,udph->len - 12, sizeof(router_domain3)-1) 
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain4,udph->len - 12, sizeof(router_domain4)-1) 
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain5,udph->len - 12, sizeof(router_domain5)-1) 
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain6,udph->len - 12, sizeof(router_domain6)-1) 
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain7,udph->len - 12, sizeof(router_domain7)-1) 
#if 0
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain2,udph->len - 12, sizeof(router_domain2)-1)
	|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain3,udph->len - 12, sizeof(router_domain3)-1)
	||findstring((unsigned char *)((unsigned char *)udph + 12),router_domain4,udph->len - 12, sizeof(router_domain4)-1)
	||findstring((unsigned char *)((unsigned char *)udph + 12),router_domain5,udph->len - 12, sizeof(router_domain5)-1)
#endif
	)
	{
		br_ipv6_filter_dns_packet(skb,DNS_TTL,(type == htons(0x001c)) ? 1:0); //construct the dns replay packet
		return 0;
	}
	
	return 1;
    		
}
#endif
void ipv6_dns_debug(struct sk_buff *skb)
{
	struct ipv6hdr *hdr;
	struct udphdr    *udph;
	struct ethhdr    *eth = (struct ethhdr *)skb->mac_header;
	  char *buf = skb->data;
         int len = skb->len;
         int i;
	
	if (eth->h_proto != htons(ETH_P_IP))
	{
		if(eth->h_proto == htons(ETH_P_IPV6))
		{
			hdr = (struct ipv6hdr *) skb->data;
			if(hdr->nexthdr == IPPROTO_UDP)
			{
				udph = (struct udphdr *)(((unsigned char *)hdr) + sizeof(struct ipv6hdr));

				if( udph->dest == htons(53) )
				{
				             printk("[%s:%d]Packet length = %#4x\n", __FUNCTION__, __LINE__, len);
				             for (i = 0; i < len; i++){
				                     if (i % 16 == 0) printk("%#4.4x", i);
				                     if (i % 2 == 0) printk(" ");
				                     printk("%2.2x", ((unsigned char *)buf)[i]);
				                    if (i % 16 == 15) printk("\n");
												}
				             printk("\n\n\n\n");
				}
			}
		}
		
	}	
}
int br_dns_netgear(struct sk_buff *skb, struct net_bridge_port *p)
{
	struct ethhdr                   *eth = (struct ethhdr *)skb->mac_header;
	struct iphdr                    *iph;
	struct udphdr                   *udph;
	if (eth->h_proto != htons(ETH_P_IP))
	{	
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
		if(eth->h_proto == htons(ETH_P_IPV6))
		{
			return br_ipv6_dns_netgear(skb);
		}
#endif
		return 1;
	}	
	/* Check for possible (maliciously) malformed IP frame (thanks Dave) */
	iph = (struct iphdr *) skb->data;
	if(iph->protocol != 17)
		return 1;
	udph = (struct udphdr *)(((unsigned char *)iph) + (iph->ihl<<2));

	if( udph->dest == htons(53) )// the packet is dns packet
	{
		unsigned char *ptmp = (unsigned char *)udph + sizeof(struct udphdr);
    	struct dns_header *dns_hdr = (struct dns_header *)ptmp;
    	
    	if ((dns_hdr->flags & htons(0x8000)) == 0 && dns_hdr->qdcount == htons(1)) //flags is query, and the query count is equal 1
    	{
    		unsigned short type = 0;
			int querySize = 0;
    		ptmp = (unsigned char *)dns_hdr + sizeof(struct dns_header);
			querySize = dns_query_name_sizeof(ptmp);
    		ptmp += querySize;
    		type = *((unsigned short *)ptmp);	//get the  query type	

		//DNSʽʾķDNSĺԴ
    		if(ptmp + 4 > skb->tail)
    		{
			/*
    			if(printk_ratelimit())
    			{
    				printk("Error DNS packet!\n");
				}
			*/
				return 1;
    		}
			/* 
			AAAAͲѯ֧֣ӦĿֹֹ
			AAAAͲѯ
			*/
    		if(type == htons(0x0001) || type == htons(0x001c))//"Host Address"query
    		{
#if 0
				char domain[15], domain2[15], domain3[17]; //fixed the 'code check' problem
				domain[0] = 0x09;
				domain2[0] = 0x09;
				domain3[0] = 0x0b; // it is the length of 'routerlogin'
				memcpy(domain + 1, "mywifiext.com",13);
				memcpy(domain2 + 1, "mywifiext.net",13);
				memcpy(domain3 + 1, "routerlogin.net",15); //add force redirect domain name for password recovery relogin function
				/*www.mywifiext.com .*/
				domain[10] = 0x03;
				domain[14]= '\0';
				domain2[10] = 0x03;
				domain2[14]= '\0';
				domain3[12] = 0x03; //length of 'net' is 0x03
				domain3[16] = '\0';
#endif				
				if(findstring((unsigned char *)((unsigned char *)udph + 12),xbox_domain0,ntohs(udph->len) - 12, sizeof(xbox_domain0)-1)
					||findstring((unsigned char *)((unsigned char *)udph + 12),xbox_domain1,ntohs(udph->len) - 12, sizeof(xbox_domain1)-1)
					||findstring((unsigned char *)((unsigned char *)udph + 12),xbox_domain2,ntohs(udph->len) - 12, sizeof(xbox_domain2)-1)
					/*WDS Repeater Modeٳֹ*/
					|| (WDS_REPEATER&sysctl_dns_hijack))
				{
					//صҪ
					return 1;
				}
				/*жǷΪBlank Stateδ״̬*/
				else if (BLANK_STATE&sysctl_dns_hijack)
				{
					br_filter_dns_packet(skb,p,0); //construct the dns replay packet
					return 0;
				}
				//ұӦDNSѯЧʱΪһ
				else if(findstring((unsigned char *)((unsigned char *)udph + 12),router_domain0,ntohs(udph->len) - 12, sizeof(router_domain0)-1)
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain1,ntohs(udph->len) - 12, sizeof(router_domain1)-1) 
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain2,ntohs(udph->len) - 12, sizeof(router_domain2)-1) 
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain3,ntohs(udph->len) - 12, sizeof(router_domain3)-1) 
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain4,ntohs(udph->len) - 12, sizeof(router_domain4)-1) 
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain5,ntohs(udph->len) - 12, sizeof(router_domain5)-1) 
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain6,ntohs(udph->len) - 12, sizeof(router_domain6)-1) 
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain7,ntohs(udph->len) - 12, sizeof(router_domain7)-1) 
#if 0
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain2,udph->len - 12, sizeof(router_domain2)-1)
						|| findstring((unsigned char *)((unsigned char *)udph + 12),router_domain3,udph->len - 12, sizeof(router_domain3)-1)
						||findstring((unsigned char *)((unsigned char *)udph + 12),router_domain4,udph->len - 12, sizeof(router_domain4)-1)
						||findstring((unsigned char *)((unsigned char *)udph + 12),router_domain5,udph->len - 12, sizeof(router_domain5)-1)
#endif
				)
				{
					br_filter_dns_packet(skb, p, DNS_TTL); //construct the dns replay packet
					return 0;
				}
				else
				{
					return 1;
				}
    		}

    	}
	}
	return 1;
}
#endif


static int br_pass_frame_up(struct sk_buff *skb)
{
	struct net_device *indev, *brdev = BR_INPUT_SKB_CB(skb)->brdev;
	struct net_bridge *br = netdev_priv(brdev);
	struct br_cpu_netstats *brstats = this_cpu_ptr(br->stats);

	u64_stats_update_begin(&brstats->syncp);
	brstats->rx_packets++;
	brstats->rx_bytes += skb->len;
	u64_stats_update_end(&brstats->syncp);

	indev = skb->dev;
	skb->dev = brdev;

	return NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN, skb, indev, NULL,
		       netif_receive_skb);
}

/* note: already called with rcu_read_lock */
int br_handle_frame_finish(struct sk_buff *skb)
{
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);
	struct net_bridge *br;
	struct net_bridge_fdb_entry *dst;
	struct net_bridge_mdb_entry *mdst;
	struct sk_buff *skb2;

	if (!p || p->state == BR_STATE_DISABLED)
		goto drop;

	/* insert into forwarding database after filtering to avoid spoofing */
	br = p->br;
#ifdef HNDCTF
	br_fdb_update(br, p, eth_hdr(skb)->h_source, skb);
#else
	br_fdb_update(br, p, eth_hdr(skb)->h_source);
#endif

	if (is_multicast_ether_addr(dest) &&
	    br_multicast_rcv(br, p, skb))
		goto drop;

	if (p->state == BR_STATE_LEARNING)
		goto drop;

	BR_INPUT_SKB_CB(skb)->brdev = br->dev;

	/* The packet skb2 goes to the local host (NULL to skip). */
	skb2 = NULL;

	if (br->dev->flags & IFF_PROMISC)
		skb2 = skb;

	dst = NULL;

	if (is_multicast_ether_addr(dest)) {
		mdst = br_mdb_get(br, skb);
		if (mdst || BR_INPUT_SKB_CB_MROUTERS_ONLY(skb)) {
			if ((mdst && mdst->mglist) ||
			    br_multicast_is_router(br))
				skb2 = skb;
			br_multicast_forward(mdst, skb, skb2);
			skb = NULL;
			if (!skb2)
				goto out;
		} else
			skb2 = skb;

		br->dev->stats.multicast++;
	} else if (((dst = __br_fdb_get(br, dest)) && dst->is_local) ||
#ifdef BCM_GMAC3
		(htons(skb->protocol) == 0x88c7) ||
		/* 0x88c7 is EAPOL Preauth frame, and for gmac3_enable/ATLAS configuration
		 * Platforms,we are allowing this packet to reach upto EAPD-> NAS, as bridge
		 * does not have the radio interfaces in it's port list.
		 */
#endif
		0) {
		skb2 = skb;
		/* Do not forward the packet since it's local. */
		skb = NULL;
	}

	if (skb) {
		if (dst)
			br_forward(dst->dst, skb, skb2);
		else
			br_flood_forward(br, skb, skb2);
	}

	if (skb2)
		return br_pass_frame_up(skb2);

out:
	return 0;
drop:
	kfree_skb(skb);
	goto out;
}

/* note: already called with rcu_read_lock */
static int br_handle_local_finish(struct sk_buff *skb)
{
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);

#ifdef HNDCTF
	br_fdb_update(p->br, p, eth_hdr(skb)->h_source, skb);
#else
	br_fdb_update(p->br, p, eth_hdr(skb)->h_source);
#endif
	return 0;	 /* process further */
}

/* Does address match the link local multicast address.
 * 01:80:c2:00:00:0X
 */
static inline int is_link_local(const unsigned char *dest)
{
	__be16 *a = (__be16 *)dest;
	static const __be16 *b = (const __be16 *)br_group_address;
	static const __be16 m = cpu_to_be16(0xfff0);

	return ((a[0] ^ b[0]) | (a[1] ^ b[1]) | ((a[2] ^ b[2]) & m)) == 0;
}
#if 1
/*
1DHCPDISCOVER0x01ΪClientʼDHCP̵ĵһ
2DHCPOFFER0x02ΪServerDHCPDISCOVERĵӦ
3DHCPREQUEST0x03˱SlientʼDHCPжserverDHCPOFFERĵĻӦclientIPַʱı
4DHCPDECLINE0x04ClientServerIPַ޷ʹãIPַͻʱ˱ģ֪ͨServerֹʹIPַ
5DHCPACK0x05ServerClientDHCPREQUESTĵȷӦģClientյ˱ĺ󣬲IPַصϢ
6DHCPNAK0x06ServerClientDHCPREQUESTĵľܾӦģClientյ˱ĺһ¿ʼµDHCP̡
7DHCPRELEASE0x07ClientͷserverIPַıģServerյ˱ĺ󣬾ͿԻIPַܹClient
8DHCPINFORM0x08ClientѾIPַʹ˱ģֻΪ˴DHCP SERVERȡһЩϢroute ipDNS IpȣֱĵӦ÷ǳټ
*/

#define DHCP_PKT_TYPE_DISC     (0x1)
#define DHCP_PKT_TYPE_OFFER    (0x2)
#define DHCP_PKT_TYPE_REQUEST  (0x3)
#define DHCP_PKT_TYPE_DECLINE  (0x4)
#define DHCP_PKT_TYPE_ACK      (0x5)
#define DHCP_PKT_TYPE_NACK     (0x6)
#define DHCP_PKT_TYPE_RELEASE  (0x7)
#define DHCP_PKT_TYPE_INFORM   (0x8)
#define OPT_CODE 0
#define OPT_LEN 1
#define OPT_DATA 2
#define OPTION_FIELD		0
#define FILE_FIELD		1
#define SNAME_FIELD		2
/* DHCP option codes (partial list) */
#define DHCP_PADDING		0x00
#define DHCP_SUBNET		0x01
#define DHCP_TIME_OFFSET	0x02
#define DHCP_ROUTER		0x03
#define DHCP_TIME_SERVER	0x04
#define DHCP_NAME_SERVER	0x05
#define DHCP_DNS_SERVER		0x06
#define DHCP_LOG_SERVER		0x07
#define DHCP_COOKIE_SERVER	0x08
#define DHCP_LPR_SERVER		0x09
#define DHCP_HOST_NAME		0x0c
#define DHCP_BOOT_SIZE		0x0d
#define DHCP_DOMAIN_NAME	0x0f
#define DHCP_SWAP_SERVER	0x10
#define DHCP_ROOT_PATH		0x11
#define DHCP_IP_TTL		0x17
#define DHCP_MTU		0x1a
#define DHCP_BROADCAST		0x1c
#define DHCP_STATIC_ROUTE	0x21 
#define DHCP_NTP_SERVER		0x2a
#define DHCP_WINS_SERVER	0x2c
#define DHCP_REQUESTED_IP	0x32
#define DHCP_LEASE_TIME		0x33
#define DHCP_OPTION_OVER	0x34
#define DHCP_MESSAGE_TYPE	0x35
#define DHCP_SERVER_ID		0x36
#define DHCP_PARAM_REQ		0x37
#define DHCP_MESSAGE		0x38
#define DHCP_MAX_SIZE		0x39
#define DHCP_T1			0x3a
#define DHCP_T2			0x3b
#define DHCP_VENDOR		0x3c
#define DHCP_CLIENT_ID		0x3d
#define DHCP_NETBIOS_NODETYPE 0x2e
#define DHCP_NETBIOS_SCOPE 0x2F
#define DHCP_END		0xFF

struct dhcpMessage {
	unsigned char op;
	unsigned char htype;
	unsigned char hlen;
	unsigned char hops;
	unsigned int xid;
	unsigned short secs;
	unsigned short flags;
	unsigned int ciaddr;
	unsigned int yiaddr;
	unsigned int siaddr;
	unsigned int giaddr;
	unsigned char chaddr[16];
	unsigned char sname[64];
	unsigned char file[128];
	unsigned int cookie;
	unsigned char options[308]; /* 312 - cookie */ 
};
struct udp_dhcp_packet {
	struct iphdr ip;
	struct udphdr udp;
	struct dhcpMessage data;
};
extern char g_szdhcp_server_interface[32] ;

#ifdef CONFIG_DHCP_MON
typedef struct device_list_table
{
	unsigned char ip[4];
	unsigned char mac[6];
	unsigned char hostname[64];
}device_list_table;

extern int g_dhcpmon_pid;
extern struct sock *netlink_dhcpmon_sock;
rwlock_t  netlink_lock;
#endif
/*
fuction: get dhcp option from udp data and option code
input: udp data, dhcp option code
return:code offset at skb
*/
unsigned char *br_get_dhcp_option(struct dhcpMessage *packet, int code, unsigned char *option_len)
{
	int i, length;
	unsigned char *optionptr = NULL;
	int over = 0, done = 0, curr = OPTION_FIELD;
	
	optionptr = packet->options;
	i = 0;
	length = 308;
	while (!done) 
    {
		if (i >= length) 
        {
			return NULL;
		}

		if (optionptr[i + OPT_CODE] == code) 
        {
			if (i + 1 + optionptr[i + OPT_LEN] >= length) 
            {
				return NULL;
			}
			*option_len = optionptr[i + OPT_LEN];
			return optionptr + i + 2;
		}			
		switch (optionptr[i + OPT_CODE]) 
        {
    		case DHCP_PADDING:
    			i++;
    			break;
    		case DHCP_OPTION_OVER:
    			if (i + 1 + optionptr[i + OPT_LEN] >= length) 
                {
    				return NULL;
    			}
    			over = optionptr[i + 3];
    			i += optionptr[OPT_LEN] + 2;
    			break;
    		case DHCP_END:
    			if (curr == OPTION_FIELD && over & FILE_FIELD) 
                {
    				optionptr = packet->file;
    				i = 0;
    				length = 128;
    				curr = FILE_FIELD;
    			} 
                else if (curr == FILE_FIELD && over & SNAME_FIELD) 
    		    {
    				optionptr = packet->sname;
    				i = 0;
    				length = 64;
    				curr = SNAME_FIELD;
    			} 
                else
                {
                    done = 1;
                }
    			break;
    		default:
    			i += optionptr[OPT_LEN + i] + 2;
		}
	}
	return NULL;
}

#ifdef CONFIG_DHCP_MON
/* ͨnetlinkʽϢӦò */
static void send_msg_to_user(struct device_list_table device_msg)
{
	int size = 0, ret = 0;
	struct device_list_table *s_device = NULL;
	unsigned char *old_tail = NULL;
	struct sk_buff *skb = NULL;
	struct nlmsghdr *nlh = NULL;

	if(g_dhcpmon_pid == 0)
	{
		printk("dhcpmon_pid's pid = %d.Please run dhcpmon_pid.\n", g_dhcpmon_pid);
	}

	size = NLMSG_SPACE(256);

	skb = alloc_skb(size, GFP_ATOMIC);
	old_tail = skb->tail;

	/* init msg header */
	nlh = NLMSG_PUT(skb, 0, 0, NETLINK_DHCP_TBS, size-sizeof(*nlh));

	/* point to msg data area */
	s_device = NLMSG_DATA(nlh);

	/* fill data for sending */
	memset(s_device, 0, size);
	s_device->ip[0] = device_msg.ip[0];
	s_device->ip[1] = device_msg.ip[1];
	s_device->ip[2] = device_msg.ip[2];
	s_device->ip[3] = device_msg.ip[3];

	s_device->mac[0] = device_msg.mac[0];
	s_device->mac[1] = device_msg.mac[1];
	s_device->mac[2] = device_msg.mac[2];
	s_device->mac[3] = device_msg.mac[3];
	s_device->mac[4] = device_msg.mac[4];
	s_device->mac[5] = device_msg.mac[5];

	memcpy(s_device->hostname, device_msg.hostname, 64);
	
	/* get netlink msg length */
	nlh->nlmsg_len = skb->tail - old_tail;

	NETLINK_CB(skb).pid = 0;
	NETLINK_CB(skb).dst_group = 0;
	//NETLINK_CB(skb).dst_pid = blp_cmm_pid;  //by ZhangYu,kernel 2.6.21 has no member dst_pid

	/* send msg */
	read_lock_bh(&netlink_lock);
	ret = netlink_unicast(netlink_dhcpmon_sock, skb, g_dhcpmon_pid, MSG_DONTWAIT);
	read_unlock_bh(&netlink_lock);

	//printk("message send! ret=%x \n",ret);

	return;
	
nlmsg_failure:
	printk("Fail to send netlink message.\n");

	if(skb)
		kfree_skb(skb);

	return;

}
#endif
/*
fuction: br_dhcp_packet

input: udp data, dhcp option code
return:code offset at skb
*/
int br_dhcp_packet(struct sk_buff *skb, unsigned char *br_dest, unsigned char *dest)
{
	struct iphdr *iph = NULL;
	struct udphdr *udph = NULL;
	struct udp_dhcp_packet *dhcp_packet = NULL;
	unsigned char option_len = 0, *temp = NULL;
#ifdef CONFIG_DHCP_MON
	struct device_list_table g_deviceListTable;
#endif
    if(NULL == skb)
		return 0;
	
    //get iph
    iph = (struct iphdr *)skb_network_header(skb);
	
	if(NULL == iph)
	    return 0;
	
    //get udph
	udph=(void *) iph + iph->ihl*4;
	if(NULL == udph)
	    return 0;
	
    //if dhcp server packet send to me
	if(iph->protocol == IPPROTO_UDP)
    {
    	if(ntohs(udph->dest) == 68)
    	{
    		if (!compare_ether_addr(br_dest, dest))
    		{
	            //up down side packet
	            if( strstr(skb->dev->name, "eth0") || strstr(skb->dev->name, "eth1"))
	            {
	                //printk("%s(%d):skb->dev->name=%s src.ip=%x, src.dst=%x\n", __FUNCTION__, __LINE__, skb->dev->name,iph->saddr, iph->daddr);
	                dhcp_packet =(struct udp_dhcp_packet *)skb->data;
					if(NULL == dhcp_packet)
	    				return 0;
	                if (NULL != (temp = br_get_dhcp_option(&(dhcp_packet->data), DHCP_MESSAGE_TYPE, &option_len))) 
	                {
						//printk("%s(%d): temp=%d\n", __FUNCTION__, __LINE__, temp[0]);
						if(DHCP_PKT_TYPE_REQUEST == temp[0])
		                {

						}
		                else if(DHCP_PKT_TYPE_ACK == temp[0])
		                {
		                    sprintf(g_szdhcp_server_interface, "%s", skb->dev->name);
						}
						else if(DHCP_PKT_TYPE_RELEASE == temp[0])
		                {
		                     
		                }    
	                }
					
	             }
    		}
    	}
#ifdef CONFIG_DHCP_MON
		else if(ntohs(udph->dest) == 67)
		{
			dhcp_packet =(struct udp_dhcp_packet *)skb->data;
			if(NULL == dhcp_packet)
				return 0;
			if(0x01 == dhcp_packet->data.op)
			{
	            if (NULL != (temp = br_get_dhcp_option(&(dhcp_packet->data), DHCP_HOST_NAME, &option_len))) 
	            {
	            	memset(&g_deviceListTable, 0, sizeof(struct device_list_table));
					
	            	memcpy(g_deviceListTable.mac, dhcp_packet->data.chaddr, 6);
					/*
					printk("mac=%02x:%02x:%02x:%02x:%02x:%02x", g_deviceListTable.mac[0], g_deviceListTable.mac[1],
						g_deviceListTable.mac[2], g_deviceListTable.mac[3], g_deviceListTable.mac[04], g_deviceListTable.mac[5]);
					*/
					if(option_len < 64)
					{
						memcpy(g_deviceListTable.hostname, temp, option_len);
						/*
						printk(" hostname=%s", g_deviceListTable.hostname);
						*/
					}

					if (NULL != (temp = br_get_dhcp_option(&(dhcp_packet->data), DHCP_REQUESTED_IP, &option_len))) 
					{
						memcpy(g_deviceListTable.ip, temp, 4);
						/*
						printk(" ip=%d.%d.%d.%d\n", g_deviceListTable.ip[0], g_deviceListTable.ip[1],
							g_deviceListTable.ip[2], g_deviceListTable.ip[3]);
						*/
					}
					send_msg_to_user(g_deviceListTable);
	            }
			}
		}
#endif
	}	
					
	return 0;
}

#endif
/*
 * Return NULL if skb is handled
 * note: already called with rcu_read_lock
 */
struct sk_buff *br_handle_frame(struct sk_buff *skb)
{
	struct net_bridge_port *p;
	const unsigned char *dest = eth_hdr(skb)->h_dest;
	int (*rhook)(struct sk_buff *skb);	
	if (skb->pkt_type == PACKET_LOOPBACK)
		return skb;
	if (!is_valid_ether_addr(eth_hdr(skb)->h_source))
		goto drop;
	skb = skb_share_check(skb, GFP_ATOMIC);
	if (!skb)
		return NULL;
	p = br_port_get_rcu(skb->dev);
    
    br_dhcp_packet(skb, p->br->dev->dev_addr, dest);
		
	if (unlikely(is_link_local(dest))) {
		/* Pause frames shouldn't be passed up by driver anyway */
		if (skb->protocol == htons(ETH_P_PAUSE))
			goto drop;

		/* If STP is turned off, then forward */
		if (p->br->stp_enabled == BR_NO_STP && dest[5] == 0)
			goto forward;
		if (NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN, skb, skb->dev,
			    NULL, br_handle_local_finish))
			return NULL;	/* frame consumed by filter */
		else
			return skb;	/* continue processing */
	}

forward:
	
	switch (p->state) {
	case BR_STATE_FORWARDING:
#ifdef CONFIG_DNS_HTTP_HIJACK
		/*add by wyh at 2014-04-22 hijack routerlogin dns packet*/
		if(br_dns_netgear(skb, p) == 0)
		{
				goto drop;
		}
#endif
		rhook = rcu_dereference(br_should_route_hook);
		if (rhook != NULL) {
			if (rhook(skb))
				return skb;
			dest = eth_hdr(skb)->h_dest;
		}
		/* fall through */
	case BR_STATE_LEARNING:
		if (!compare_ether_addr(p->br->dev->dev_addr, dest))
			skb->pkt_type = PACKET_HOST;

		NF_HOOK(NFPROTO_BRIDGE, NF_BR_PRE_ROUTING, skb, skb->dev, NULL,
			br_handle_frame_finish);
		break;
	default:
drop:
		kfree_skb(skb);
	}
	return NULL;
}
